import ui, dialogs, sound
import os, math
import builtins

class RollView(ui.View):
	rollsPerFace = 10
	emptyFilename = 'untitled die'

	def did_load(self):
		self.grid = Grid(items=[])
		self['progress'].hidden = True
		self['dieRolls'].data_source = self.grid
		self['dieRolls'].delegate = self.grid
		self['dieSize'].action = self.chooseDieSize
		self['load'].action = self.load
		self['trash'].action = self.trash
		self['share'].action = self.share
		self['dieName'].delegate = self
		self.setDie()

	def draw(self):
		if hasattr(self, 'die'):
			self.createGrid()

	def setDie(self, die='d20'):
		self['dieSize'].title = die
		self.die = int(die[1:])
		self.rollCount = 0
		self.rolls = {}
		self.filename = None
		self['dieName'].text = self.emptyFilename
		self.createGrid()

	def createGrid(self):
		values = range(1, self.die+1)
		rowHeight = self['dieRolls'].height/self.die
		self['dieRolls'].row_height = rowHeight
		self['dieRolls'].height = self.die*rowHeight
		self.grid.font = ('<system>', rowHeight*.4)
		self.grid.items = values

		self.progress()

	def progress(self):
		if not self.rollCount:
			self['progress'].hidden = True
			return
		self['progress'].hidden = False

		maxWidth = self['progressGroove'].width
		sampleSize = self.sampleSize()
		self['progress'].width = min(sampleSize, self.rollCount)/sampleSize*maxWidth

		if self.finished():
			self['progress'].text, results = self.chisquare()
		else:
			self['progress'].text = str(sampleSize-self.rollCount)
			self['progress'].background_color = '#25a525'

	#there should be at least fifty rolls,
	#and enough rolls that most results are hit at least five times
	def sampleSize(self):
		return max(50, self.die*self.rollsPerFace)

	def getDieProgress(self, roll):
		if not roll in self.rolls:
			return None, None

		count = self.rolls[roll]
		maximum = max(self.rollsPerFace*2, max(self.rolls.values()))
		return maximum, count

	def chooseDieSize(self, button):
		if self.dirty():
			response = dialogs.alert('Unsaved data', 'Your current data is not saved. if you start a new die you will lose those rolls.', 'Start new die')

		dice = ['d4', 'd6', 'd8', 'd10', 'd12', 'd20']
		die = dialogs.list_dialog('Choose a die type', dice)
		if not die:
			dialogs.hud_alert('New die cancelled.', 'error')
			return

		self.setDie(die)

	#backgrounding is necessary to keep alert from freezing pythonista
	@ui.in_background
	def textfield_did_end_editing(self, textview):
		textview.text = textview.text.strip()
		if textview.text == self.emptyFilename:
			return
		if textview.text == self.filename:
			return
		if not textview.text:
			textview.text = self.emptyFilename
			return

		filepath = self.makeFilepath(textview.text)
		if os.path.exists(filepath):
			response = dialogs.alert('File exists', 'The file ' + textview.text + ' already exists. Do you want to erase that file?', 'Yes, erase', 'No, cancel', hide_cancel_button=True)
			if response == 2:
				textview.text = self.filename or self.emptyFilename
				return
		self.filename = textview.text
		self.save()

	def makeFilepath(self, filename=None):
		if not filename:
			filename = self.filename
		return os.path.join('data', filename) + '.diesquare'

	def finished(self):
		return self.rollCount >= self.sampleSize()

	def dirty(self):
		return self.rolls and not self.filename

	def save(self):
		if not self.filename:
			return
		if not os.path.exists('data'):
			os.mkdir('data')

		filepath = self.makeFilepath()
		file = open(filepath, 'w')
		file.write('d' + str(self.die) + "\n")
		for roll in range(1, self.die+1):
			if roll in self.rolls:
				count = self.rolls[roll]
			else:
				count = 0
			file.write(str(roll) + ': ' + str(count) + "\n")
		file.close()

	def load(self, button):
		if self.dirty():
			response = dialogs.alert('Unsaved data', 'Your current data is not saved. if you load a new file you will lose those rolls.', 'Load new file')

		files = None
		if os.path.exists('data'):
			files = os.listdir('data')
		if not files:
			dialogs.hud_alert('There are no files to load.', 'error')
			return

		files = [os.path.splitext(file)[0] for file in files]
		file = dialogs.list_dialog('Choose file to load:', files)
		if not file:
			dialogs.hud_alert('File load cancelled.', 'error')
			return

		data = open(self.makeFilepath(file)).readlines()
		die = data.pop(0)
		if not die.startswith('d'):
			dialogs.hud_alert('Bad data', 'error')
			return

		self.setDie(die)
		self.filename = file
		self['dieName'].text = file
		for roll in data:
			roll = roll.strip()
			roll, count = roll.split(': ')
			roll, count = int(roll), int(count)
			if count:
				self.rolls[roll] = count
				self.rollCount += count

		self.progress()
		self['dieRolls'].reload()

	def trash(self, button):
		if self.rolls:
			if self.filename:
				dialogs.alert('Erase file?', 'Erase this file and start over?', 'Yes, erase')
			else:
				dialogs.alert('Clear data?', 'Lose this data and start over?', 'Yes, erase')

		self.rolls = {}
		self.rollCount = 0
		self['dieRolls'].reload()
		self.progress()

	def share(self, button):
		options = ['Text Table', 'R Script']
		if self.rollCount:
			options.append('Chi-Square Calculations')
		format = dialogs.list_dialog('Choose format to share:', options)
		if format == 'R Script':
			die = str(self.die)
			observed = 'd' + die + ' <- matrix(c('
			numbers = []
			for roll in range(1, self.die+1):
				if roll in self.rolls:
					count = self.rolls[roll]
				else:
					count = 0
				numbers.append(str(count))
			observed += ', '.join(numbers)
			observed += '), ncol=' + die + ', byrow=TRUE)'
			expected = 'probability = rep(1/' + die + ', ' + die + ')'
			chisquare = 'chisq.test(d' + die + ', p=probability)'
			lines = [observed, expected, chisquare]
		elif format == 'Text Table':
			lines = []
			if self.filename:
				lines.append('# ' + self.filename)
				lines.append('')
			lines.append('## d{} ({} rolls)'.format(self.die, self.rollCount))
			lines.append('')
			lines.append("roll\tcount")
			for roll in range(1, self.die+1):
				if roll in self.rolls:
					count = self.rolls[roll]
				else:
					count = 0
				lines.append(str(roll) + '\t' + str(count))
		elif format == 'Chi-Square Calculations':
			pvalue, lines = self.chisquare()
		else:
			return
		text = "\n".join(lines)
		destination, dummy = dialogs.share_text(text)
		if destination:
			dialogs.hud_alert('Shared!', 'success')
		else:
			dialogs.hud_alert('Share canceled.', 'error')

	def newRoll(self, roll):
		if self.finished():
			sound.play_effect('game:Error')
			return

		if roll in self.rolls:
			self.rolls[roll] += 1
		else:
			self.rolls[roll] = 1
		self.rollCount += 1
		self.progress()
		pitch = roll/self.die + .4
		sound.play_effect('arcade:Coin_4', pitch=pitch)
		self.save()

	def removeRoll(self, roll):
		if roll in self.rolls:
			if self.rolls[roll] > 0:
				self.rolls[roll] -= 1
				self.rollCount -= 1
				self.progress()
				sound.play_effect('arcade:Explosion_1')
				self.save()
				return

		sound.play_effect('game:Error')

	#http://specminor.org/2017/01/08/performing-chi-squared-gof-python.html
	#http://www.statisticshowto.com/probability-and-statistics/chi-square/
	def chisquare(self):
		results = ['\n# {} d{}'.format(self.filename, self.die)]
		expected = self.rollCount/self.die
		chisquare = 0
		for roll in range(1, self.die+1):
			if roll in self.rolls:
				observed = self.rolls[roll]
			else:
				observed = 0
			component = (observed - expected)**2/expected
			chisquare += component

		color = 'orange'
		pvalue = 0
		freedom = self.die - 1
		cdf = self.cdf(freedom/2, chisquare/2)
		if cdf != False:
			gamma = self.gamma(freedom/2)
			if gamma != False:
				pvalue = 1 - cdf/gamma

				results.append('cdf: {}; gamma: {}'.format(cdf, gamma))
				results.append('freedom: {}; X-square: {}'.format(freedom, chisquare))
				results.append('p-value: {}'.format(pvalue))

				if pvalue < 0 or pvalue > 1:
					color = 'orange'
					result = 'invalid p-value'
				elif pvalue < .05:
					if pvalue < .01:
						color = 'red'
						result = 'probably biased'
					else:
						color = 'yellow'
						result = 'possibly biased'
				else:
					result = 'bias unlikely'
					color = '#25a525'
			else:
				result = 'invalid gamma'
		else:
			result = 'invalid cdf'
		results.append(result)

		self['progress'].background_color = color
		return 'p = {:.4g} ({})'.format(pvalue, result), results

	#gamma function?
	def gamma(self, x):
		#Play with these values to adjust the error of the approximation.
		upper_bound = 100
		resolution = 1000000

		step = upper_bound/resolution

		value = 0
		rollingSum = 0

		#this gets rounded to 2.718281828459045
		e = 2.7182818284590452353602874713526624977
		previous = None
		while value <= upper_bound:
			rollingSum += step*(value**(x-1)*e**(-value))
			value += step
			if rollingSum == previous:
				previous = None
				break
			previous = rollingSum

		if previous:
			return False
		else:
			return rollingSum

	#cumulative distribution function?
	def cdf(self, s, z):
		rollingSum = 0
		previous = None
		for k in range(0, 200):
			try:
				rollingSum += (((-1)**k)*z**(s+k))/(math.factorial(k)*(s+k))
			except OverflowError:
				break

			if previous == rollingSum:
				previous = None
				break
			previous = rollingSum

		if previous:
			return False
		else:
			return rollingSum

class Grid(ui.ListDataSource):
	def tableview_did_select(self, tableview, section, row):
		roll = self.items[row]
		view.newRoll(roll)
		tableview.selected_row = 0,-1
		tableview.reload()

	def tableview_delete(self, tableview, section, row):
		roll = self.items[row]
		view.removeRoll(roll)
		tableview.reload()
		return False

	def tableview_cell_for_row(self, tableview, section, row):
		rowView = ui.TableViewCell()
		roll = self.items[row]

		# create cell for roll number
		rollCell = ui.Label(text = str(roll))
		rollCell.height = tableview.row_height - 4
		rollCell.font = self.font
		rollCell.y = 2
		rollCell.width = 1.5*self.font[1]
		rollCell.alignment = ui.ALIGN_RIGHT
		rowView.add_subview(rollCell)

		# create cell for count bar
		maximumUnits, units = view.getDieProgress(roll)
		if units:
			progress = ui.Label(text = str(units), alignment = ui.ALIGN_CENTER)
			progress.x = 2.2*self.font[1]
			progress.height = tableview.row_height-4
			progress.y = 2
			progress.width = units/maximumUnits*(tableview.width - progress.x)
			progress.background_color = '#2f6bff'
			rowView.add_subview(progress)

		return rowView

try:
	view = builtins.dieView
except AttributeError:
	view = None

if not view or not view.on_screen:
	view = ui.load_view()
	builtins.dieView = view
	view.present()

