import datetime, shelve
import makeHTML
from snakeserver.snakelet import Snakelet

#the name of the RSS file, change as desired
#URL will be http://host:port/blog/{RSSPage}
RSSPage = "RSS"

#the name of the "copy to my blog" file; will produce entries suitable for copying the HTML to another blog
Blogger = "copy"
bloggerStartLevel = 3

#the name of the main page--Snakelet sets this, we need to recognize it
#this will be the page that we create an index on rather than a post
indexPage = "index.sn"

#The Blog mediates between Posts and Page
class Blog(Snakelet):
	def __init__(self, request, response):
		Snakelet.__init__(self, request, response)
		self.app = self.getAppContext()

		#set up the posts and pages if we don't have any yet
		if not hasattr(self.app, "posts"):
			self.app.posts = Posts()

		#if this page is called again, most likely we will need to regenerate the main page
		#so always set pages to nothing
		self.app.pages = {}

	#if canEdit returns True, this user will get forms for non-existent URLs
	#and posted info will be looked at and stored
	#currently checks that the server and the client are the same IP
	def canEdit(self, request):
		if request.getServerIP() == request.getRemoteAddr():
			print "Remote:", request.getServerIP(), "-", "Local:", request.getRemoteAddr()
			return True
		else:
			print "Tried to get admin priv:", request.getRemoteAddr()
			return False

	#pull title and description from main post
	def getTitle(self):
		if self.app.posts.hasPost(indexPage):
			main = self.app.posts.getPost(indexPage)
			return main.getTitle()
		else:
			return None

	def getDescription(self):
		if self.app.posts.hasPost(indexPage):
			main = self.app.posts.getPost(indexPage)
			return main.getDescription()
		else:
			return None
	
	#find the correct item to display in response to a request
	#current options include a post, a form, or an RSS feed
	def getPage(self, request, response):
		paths = request.getRequestURL().split('/')
		path = paths[2]

		if request.getField("Title") and self.canEdit(request):
			#create new post from form response
			self.app.posts.createPost(path, request)
			self.recreateIndex()

		if path==RSSPage:
			response.setContentType("application/rss+xml")

		if not path in self.app.pages:
			if self.app.posts.hasPost(path):
				#create a page from an existing post
				post = self.app.posts.getPost(path)
				self.app.pages[path] = Page(path, self.app.posts, post, context=self.getWebApp())
			elif path==RSSPage:
				#self.app.pages[path] = self.RSS(request, response)
				home = self.app.posts.getPost(indexPage)
				self.app.pages[path] = Page(path, self.app.posts, home, context=self.getWebApp(), request=request)
			elif path==Blogger:
				home = self.app.posts.getPost(indexPage)
				self.app.pages[path] = Page(path, self.app.posts, home, context=self.getWebApp())
			elif self.canEdit(request):
				#we need a form to enter a new post
				return Page(path,self.app.posts, context=self.getWebApp())
			else:
				response.sendError(404, "No Sense of Post: " + path)
				return None

		return self.app.pages[path]

	#recreate the various indexes of pages
	def recreateIndex(self):
		if indexPage in self.app.pages:
			post = self.app.posts.getPost(indexPage)
			self.app.pages[indexPage] = Page(indexPage, self.app.posts, post, context=self.getWebApp())

		#remove these so that they will get recreated
		if RSSPage in self.app.pages:
			del self.app.pages[RSSPage]
		if Blogger in self.app.pages:
			del self.app.pages[Blogger]

	#respond to the browser's request
	def serve(self, request, response):
		#handle this visit
		page = self.getPage(request, response)

		if page:
			out=response.getOutput()
			out.write(page)

#A post is a single item
#It knows how to create its own HTML and RSS
#if we add new features later, we need to use getattr or hasattr because previous ones
#will not exist in earlier objects
class Post:
	def __init__(self, request):
		self.title = request.getField("Title")
		self.body = request.getField("body")
		self.link = request.getField("link")
		self.stamp = datetime.datetime.now()

	def getTitle(self):
		return getattr(self, "title", None)

	def getLink(self):
		return getattr(self, "link", None)

	def getDescription(self):
		description = getattr(self, "body", None)
		if not description:
			description = self.getTitle()
		return description

	def RSS(self, path):
		item = makeHTML.part("item")
		item.addPart("title", self.getTitle())
		item.addPart("link", str(path))
		item.addPart("description", self.getDescription())
		item.addPart("pubDate", self.getPubDate())
		
		return item

	def bodyHTML(self, style=None):
		if hasattr(self, "body"):
			parts = self.body.splitlines()
			paragraphs = []
			for part in parts:
				if part:
					paragraph = makeHTML.part("p", content=makeHTML.encode(part), style=style)
					paragraphs.append(paragraph)
			return paragraphs
		return None


	def HTML(self, level=2):
		html = makeHTML.part("div", style="post")
		title = makeHTML.headline(self.title, level)
		stamp = makeHTML.part("p", content=self.getStamp(), style="stamp")
		piecesToAdd = [title, stamp]

		paragraphs = self.bodyHTML()
		if paragraphs:
			body = makeHTML.part("div", style="note")
			body.addPieces(paragraphs)
			piecesToAdd.append(body)
		
		if hasattr(self, "link"):
			if self.link:
				link = makeHTML.part("p", style="link")
				link.addPiece(makeHTML.link(url=self.link, content=self.link))
				piecesToAdd.append(link)
			
		html.addPieces(piecesToAdd)

		return html

	def getStamp(self):
		return self.stamp.strftime("%B %d, %Y %I:%M:%S %p")

	#this is wrong because I don't understand Python's time zones
	def getPubDate(self):
		return self.stamp.strftime("%a, %d %b %Y %H:%M:%S +0000")

#Posts will store all of the Post instances.
#It also knows how to create a new post
#and it stores the posts in a file to keep them available across server restarts.
class Posts:
	#this stores the database of posts in the main snakelets folder
	#you could also (and probably should) use a full path to anywhere you've got write access
	datafile = "posts"

	def __init__(self):
		self.posts = {}

		#read any existing posts
		datastore = shelve.open(self.datafile)
		for page in datastore:
			self.posts[page] = datastore[page]
		datastore.close()

	def hasPost(self, page):
		return page in self.posts
	
	def getPages(self):
		pages = self.posts.keys()
		pages.sort(lambda x,y:cmp(self.posts[y].stamp, self.posts[x].stamp))

		return pages

	def getPost(self, page):
		if page in self.posts:
			return self.posts[page]
		else:
			return None
	
	#generate an HTMLlist of all posts
	def indexHTML(self, startLevel=2, permaLinks=True):
		index = None
		if self.posts:
			index = makeHTML.part("div", style="postings")
			
			#sort posts by timestamp, descending
			pages = self.getPages()
			for page in pages:
				if page != indexPage:
					post = self.posts[page]
					posting = post.HTML(level=startLevel)
					permalink = makeHTML.part("p", style="permalink")
					permalink.addPiece(makeHTML.link(url=page, content="permalink"))
					if permaLinks:
						posting.addPiece(permalink)
					index.addPiece(posting)

		return index

	def makeRSS(self, context):
		items = []
		if self.posts:
			pages = self.getPages()
			for page in pages:
				post = self.posts[page]
				items.append(post.RSS(context.mkUrl(page)))
			
		return items

	def createPost(self, page, request):
		post = Post(request)
		self.store(page, post)

	def store(self, page, post):
		self.posts[page] = post
		datastore = shelve.open(self.datafile)
		datastore[page] = post
		datastore.close()
		print "Wrote", page, "to", self.datafile

	def newPost(self, page):
		titleText = page.replace("-", " ").replace("%20", " ")
		
		form = makeHTML.form(method="post", action=page)
		formItems = makeHTML.table(style="postform", firstColumnHeader=True)

		title = makeHTML.input("text", name="Title", value=titleText, id="title")
		#I'd prefer not to have any text in the textarea, but textareas can't self-close
		posttext = makeHTML.part("textarea", content="Type post here", attributes={"name": "body", "id": "body"})
		postlink = makeHTML.input("text", name="link", id="link")
		
		formItems.addRows([["Subject", title], ["Post", posttext], ["Link", postlink]])
		form.addPiece(formItems)
		
		return form

#The Page class creates a page based either on a post, or on a post that doesn't exist.
#It calls the Posts instance to do most of its work.
class Page:
	def __init__(self, path, posts, post=None, context=None, request=None):
		if path==Blogger:
			head, body = self.createIndex(post, posts, startLevel=bloggerStartLevel, permaLinks=False)
		elif path==RSSPage:
			feed = self.createRSS(post, posts, context, request)
			self.page = feed
			return
		elif post:
			if path == indexPage:
				head, body = self.createIndex(post, posts)
			else:
				head, body = self.createPost(post)
		else:
			head, body = self.newPostForm(path, posts)
		self.constructPage(head, body, context)

	#a link to return to the main page
	def returnLink(self):
		link = makeHTML.part("p", style="return")
		link.addPiece(makeHTML.link(url="./", content="Return to index"))
		return link

	def newPostForm(self, path, posts):
		head = makeHTML.head("Create New Post")
		body = makeHTML.body(style="newpost")

		div = makeHTML.part("div", style="newpost")
		div.addPiece(makeHTML.headline("Enter new post"))
		div.addPiece(posts.newPost(path))
		div.addPiece(self.returnLink())

		body.addPiece(div)

		return head, body

	def constructPage(self, head, body, context):
		page = makeHTML.part("html")

		host, port = context.getVirtualHost()
		rssLink = "http://" + host + ":" + str(port) + context.mkUrl(RSSPage)
		linkAttributes = {"REL":"alternate", "type":"application/rss+xml", "title":"RSS", "href":rssLink}
		page.addPiece(makeHTML.part("LINK", attributes=linkAttributes))
		
		if context:
			stylesheet = context.mkAssetUrl("css/blog")
			head.addPiece(makeHTML.styleSheet(stylesheet))

		page.addPieces([head, body])

		self.page = page.make()

	def __str__(self):
		return self.page
		
	#create the RSS feed for this collection of posts
	def createRSS(self, home, posts, context, request):
		site = context.getSnakelet('index.sn')
		feed = makeHTML.part("rss", attributes={"version":"2.0"})
		channel = makeHTML.part("channel")
		channel.addPart("title", site.getTitle())
		channel.addPart("link", str(request.getBaseURL() + context.getURLprefix()))
		channel.addPart("description", site.getDescription())
		channel.addPart("language", "en-us")
		channel.addPieces(posts.makeRSS(context))
		feed.addPiece(channel)
		
		feed = '<?xml version="1.0" ?>\n' + feed.make()
		return feed

	#create the main page
	def createIndex(self, home, posts, startLevel=1, permaLinks=True):
		head = makeHTML.head(home.title)
		body = makeHTML.body(style="index")
		div = makeHTML.part("div", style="index")
		div.addPiece(makeHTML.headline(home.title, startLevel))

		if home.body:
			div.addPieces(home.bodyHTML(style="description"))
		
		div.addPiece(posts.indexHTML(startLevel=startLevel+1, permaLinks=permaLinks))
		body.addPiece(div)
		return head, body

	def createPost(self, post):
		head = makeHTML.head(post.title)
		body = makeHTML.body(style="permalink")

		div = makeHTML.part("div", style="permalink")
		div.addPiece(post.HTML(level=1))
		div.addPiece(self.returnLink())

		body.addPiece(div)

		return head, body
