#!/usr/bin/python
# -*- coding: utf-8 -*-
# create Daredevils character sheets from source file
# Jerry Stratton godsmonsters.com/daredevils

import scribus
import glob, os, re, sys
import time

#character import options
#folder of characters
dataDir = os.path.expanduser('~/Desktop/Daredevils/data')
#should all characters in that folder be imported?
importScope = scribus.messageBox(u'Import all?', u'Import all files from ' + dataDir + u'?', button1=scribus.BUTTON_YES, button2=scribus.BUTTON_NO, button3=scribus.BUTTON_CANCEL|scribus.BUTTON_DEFAULT|scribus.BUTTON_ESCAPE)
if importScope == scribus.BUTTON_YES:
	allCharacters = True
elif importScope == scribus.BUTTON_CANCEL:
	sys.exit()
else:
	allCharacters = False

#general values
scoreLine = re.compile('([A-Z][a-zA-Z ]+):\t(.*)$')
boxes = {
	'character':    (4.9,   .625, 3.5, .27),
	'player':       (4.9,   .875, 3.5, .27),
	'nationality':  (4.9,  1.125, 3.5, .27),
	'career':       (4.9,  1.375, 3.5, .27),
	'episode':      (4.9,  1.625, 3.5, .27),

	'age':          (7.55, .625, .5,   .27),
	'height':       (7.55, .875, .5,   .27),

	'wit':      (1.7, 2.50, .3, .24),
	'will':     (1.7, 2.72, .3, .24),
	'strength': (1.7, 2.94, .3, .24),
	'deftness': (1.7, 3.16, .3, .24),
	'speed':    (1.7, 3.38, .3, .24),
	'health':   (1.7, 3.60, .3, .24),

	'charismatic':   (2.1, 4.60, .24, .24),
	'combative':     (2.1, 4.82, .24, .24),
	'communicative': (2.1, 5.04, .24, .24),
	'esthetic':      (2.1, 5.26, .24, .24),
	'mechanical':    (2.1, 5.48, .24, .24),
	'natural':       (2.1, 5.70, .24, .24),
	'scientific':    (2.1, 5.92, .24, .24),

	'combat dodge ability':    (2.8, 6.9,  .5, .24),
	'shock factor': (2.8, 7.12, .5, .24),
	'off hand dexterity':            (2.8, 7.34, .5, .24),
	'hand damage':      (2.8, 7.56, .5, .24),
	'healing rate':             (2.8, 7.78, .5, .24),
	'perception':            (2.8, 8.00, .5, .24),
	'encumbrance capacity':              (2.8, 8.22, .5, .24),
	'luck':    (2.8, 8.44, .5, .24),

	'damage resistance total': (2.8, 9.2, .5, .24),

	'skills': (4.5, 2.25, 2.8, .24),
	#this is just the width, as the rest comes from the skills value
	'bcs': .4,

	'page two': (.6, 1.1, 7.3, 2),
	#this is the height of the quotes title
	'quote title': .5
}
pageBottom = 10.5

def die(title, message):
	message = characterFile + ': ' + message
	scribus.messageBox(title, message)
	sys.exit()

#populate a Daredevils character sheet
class Sheet:
	def __init__(self, characterName):
		self.skillsRect = self.getRect('skills')
		self.backgroundItems = []
		self.quotes = []
		self.characterName = characterName
		self.openSection('aspects')

		# create the layer for this character sheet
		#it would be easier to just delete the layer if it already exists
		#but deleting layers does not delete the objects in the layer, it just moves them down a layer
		scribus.gotoPage(1)
		if characterName in scribus.getLayers():
			scribus.setActiveLayer(characterName)
		else:
			scribus.createLayer(characterName)
		self.createBox('Character', characterName)

	def openSection(self, section):
		self.section = section
		#clear out skills, because changes here might result in overlapping boxes
		if section == 'skills':
			skillBoxPrefix = self.makeBoxName()
			for box in scribus.getAllObjects(scribus.ITEMTYPE_TEXTFRAME, 0, self.characterName):
				if box.startswith(skillBoxPrefix):
					scribus.deleteObject(box)

	# create a text box at the correct location
	def createBox(self, title, value):
		item = title.lower()
		location = self.getRect(item)
		boxName = self.makeBoxName(title)
		if "\t" in value:
			values = value.split("\t")
			#values = values[1:]
			value = values.pop(0)
			self.createOrUseBox(boxName, location, value, scribus.ALIGN_RIGHT)
			column = 0
			for value in values:
				column += 1
				columnName = boxName + '_' + str(column)
				location[0] += location[2] * 1.3
				self.createOrUseBox(columnName, location, value, scribus.ALIGN_RIGHT)
		else:
			if self.section == 'calculations':
				alignment = scribus.ALIGN_RIGHT
			else:
				alignment = None
			self.createOrUseBox(boxName, location, value, alignment)

		return True

	def createOrUseBox(self, boxName, location, value, alignment=None):
		if scribus.objectExists(boxName):
			scribus.moveObjectAbs(location[0], location[1], boxName)
			scribus.sizeObject(location[2], location[3], boxName)
		else:
			scribus.createText(location[0], location[1], location[2], location[3], boxName)
		scribus.setText(value, boxName)
		if alignment:
			scribus.setTextAlignment(alignment, boxName)

	def createTalentCategory(self, talent):
		boxName = self.makeBoxName(talent)
		self.createOrUseBox(boxName, self.skillsRect, talent)
		scribus.setCharacterStyle('Row Title', boxName)
		self.skillsRect[1] += self.skillsRect[3]

	def createSkillBox(self, skillInfo):
		skill, bcs = skillInfo.split("\t")

		#the skill name and score
		skillBoxName = self.makeBoxName(skill)
		skillBoxRect = [self.skillsRect[0]+self.skillsRect[3]/2, self.skillsRect[1], self.skillsRect[2], self.skillsRect[3]]
		self.createOrUseBox(skillBoxName, skillBoxRect, skill)

		#the Base Chance of Success
		bcsBoxName = skillBoxName + '_bcs'
		bcsBoxRect = [self.skillsRect[0]+self.skillsRect[2], self.skillsRect[1], boxes['bcs'], self.skillsRect[3]]
		self.createOrUseBox(bcsBoxName, bcsBoxRect, bcs, scribus.ALIGN_RIGHT)
		self.skillsRect[1] += self.skillsRect[3]

	def createLongTextBox(self, boxName, boxRect, textItems):
		scribus.gotoPage(2)
		text = "\n".join(textItems)
		self.createOrUseBox(boxName, boxRect, text)
		scribus.setColumns(2, boxName)
		scribus.setColumnGap(.25, boxName)
		scribus.setParagraphStyle('Long Text', boxName)
		scribus.hyphenateText(boxName)

		#resize as needed
		while scribus.textOverflows(boxName):
			boxRect[3] += .2
			if boxRect[1] + boxRect[3] > pageBottom:
				die('Warning', 'There is too much long text for page two')
			scribus.sizeObject(boxRect[2], boxRect[3], boxName)

	def addBackground(self, background):
		self.backgroundItems.append(background)

	def closeBackground(self):
		if len(self.backgroundItems) > 0:
			boxRect = self.getRect('page two')
			boxName = self.makeBoxName('background_text')
			self.createLongTextBox(boxName, boxRect, self.backgroundItems)
			self.backgroundItems = []
			return boxName
		else:
			scribus.messageBox('Missing data', 'No background for ' + self.characterName)

	def addQuote(self, quote):
		self.quotes.append(quote)

	def closeQuotes(self):
		if len(self.quotes) > 0:
			boxRect = self.getRect('page two')
			backgroundPosition = scribus.getPosition(self.backgroundBox)
			backgroundSize = scribus.getSize(self.backgroundBox)
			backgroundBottom = backgroundPosition[1] + backgroundSize[1]
			boxRect[1] = backgroundBottom + .25
			boxName = self.makeBoxName('quote_text')
			self.createLongTextBox(boxName, boxRect, self.quotes)

			#move quotes to the bottom of the page
			position = scribus.getPosition(boxName)
			size = scribus.getSize(boxName)
			if position[1] + size[1] < pageBottom:
				shift = pageBottom - (position[1] + size[1])
				scribus.moveObject(0, shift, boxName)

				#insert the Quotes section title
				titleBox = self.makeBoxName('quote_title')
				titleRect = self.getRect('page two')
				titleHeight = boxes['quote title']
				titleRect[1] = position[1] + shift - titleHeight
				titleRect[3] = titleHeight
				self.createOrUseBox(titleBox, titleRect, 'Quotes', scribus.ALIGN_CENTERED)
				scribus.setCharacterStyle('Section Title', titleBox)
			else:
				scribus.messageBox('Warning', 'Not enough room for quotes title; consider drawing a line between background and quotes')
			self.quotes = []
		else:
			scribus.messageBox('Missing data', 'No quotes for ' + self.characterName)

	def closeSection(self):
		#close any sections that have been collecting text
		#in preparation for the next section
		if self.section == 'background':
			self.backgroundBox = self.closeBackground()
		elif self.section == 'quotes':
			self.closeQuotes()

	def close(self):
		if len(self.backgroundItems) > 0:
			self.closeBackground()
		elif len(self.quotes) > 0:
			self.closeQuotes()
		#avoid accidental deletions
		scribus.deselectAll()

	def makeBoxName(self, name=''):
		boxName = self.characterName + '_' + self.section + '_' + name
		boxName = boxName.lower()
		boxName = boxName.replace(' ', '_')
		boxName = boxName.replace('*', '_')
		return boxName

	def getRect(self, boxName):
		if boxName in boxes:
			return list(boxes[boxName])
		else:
			die("Unknown item", "There is no box location for " + boxName)

class Character:
	def __init__(self, file):
		characterHandle = open(file)
		self.characterInfo = characterHandle.readlines()
		characterHandle.close()

	def populate(self):
		for line in self.characterInfo:
			line = line.rstrip()
			if line == '' or line.startswith("\t"):
				continue
			elif line.startswith('# '):
				#new character
				self.characterName = line[2:]
				sheet = Sheet(self.characterName)
			elif not self.characterName:
				die('Top of character file must be the character’s name.')
			elif re.match(scoreLine, line):
				#a simple data item
				line = re.match(scoreLine, line)
				title, value = line.group(1), line.group(2)
				if title.lower() == 'height':
					value = value.replace('”', '"')
					value = value.replace('’', "'")
				sheet.createBox(title, value)
			elif line.startswith('## '):
				sheet.closeSection()
				sheet.openSection(line[3:].lower())
			elif sheet.section == 'skills':
				if line.startswith('### '):
					talent = line[4:]
					sheet.createTalentCategory(talent)
				elif "\t" in line:
					sheet.createSkillBox(line)
				else:
					die("Strange line in skill section:\n" + line)
			elif sheet.section == 'background':
				sheet.addBackground(line)
			elif sheet.section == 'quotes':
				sheet.addQuote(line)
			else:
				die("Unparsed Line", "Unable to parse line:\n" + line)

		#close out any unclosed collections
		sheet.close()

#don't do anything unless there's a document open
if scribus.haveDoc():
	os.chdir(dataDir)

	if allCharacters:
		characterFiles = sorted(glob.glob('*.txt'))
		scribus.progressReset()
		scribus.progressTotal(len(characterFiles))
		for characterFile in characterFiles:
			#statusMessage is aspirational;
			#Scribus doesn't appear to actually update it while the script is running
			#nor can I find a scribus.refresh()
			scribus.statusMessage(u'Reading ' + characterFile)
			scribus.progressSet(characterFiles.index(characterFile)+1)
			#allow for progress to update
			time.sleep(.1)
			character = Character(characterFile)
			character.populate()
		scribus.progressReset()
	else:
		characterFile = scribus.fileDialog('Character to import', filter="*.txt")
		if not characterFile:
			sys.exit()
		character = Character(characterFile)
		character.populate()

		#hide every character except this one or it gets really confusing
		for layer in scribus.getLayers():
			if layer == character.characterName:
				scribus.setLayerVisible(layer, True)
				scribus.setLayerPrintable(layer, True)
			elif not layer.startswith('Background '):
				scribus.setLayerVisible(layer, False)
				scribus.setLayerPrintable(layer, False)
else:
	scribus.messageBox("No Open Document", "You need to have a character sheet document open to create character sheets.")
